/*########################################################################################################################################################
 *
 *  determine the number of entries for both ALMA tables
 *  Copyright (C) 2012 - 2014  Thomas Moeller
 *
 *  I. Physikalisches Institut, University of Cologne
 *
 *
 *
 *
 *
 *  input variables:     querylength:                length of query
 *                       queryString1:               first part of query string
 *                       queryString2:               second part of query string
 *                       NumberOfMolecules:          number of molecules which should be considered (used for query string)
 *                       MoleculeNames:              names of molecules which should be considered (used for query string)
 *                       dbName:                     path and name database file
 *                       NameOfColumnName:           name of column containing the name
 *
 *  output variables:    ok:                         status variable
 *                       number_lines:               number of entries in the database
 *
 *
 *
 *  Versions of the program:
 *
 *  Who           When        What
 *  T. Moeller    31.05.2012  Initial version
 *  T. Moeller    22.01.2013  change from MySQL to sqlite3
 *
 *
 *
 *  License:
 *
 *    GNU GENERAL PUBLIC LICENSE
 *    Version 3, 29 June 2007
 *    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
 *
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
########################################################################################################################################################*/


/* make a connection to the database */
#include <stdlib.h>
#include <stdio.h>
#include "sqlite3.h"

#define nameLength 40

#ifdef __cplusplus
extern "C"
#endif

void numentr_sqlite3_(int *ok, int *number_lines, int querylength,                          // status variable, number of entries, length of query
                      char queryString1[], char queryString2[],                             // first and second part of query string
                      int NumberOfMolecules,                                                // number of molecules which should be considered
                      char dbName[],                                                        // name of database
                      char NameOfColumnName[],                                              // name of column containing the name
                      char MoleculeNames[NumberOfMolecules][nameLength]                     // name of molecules which should be considered
                      )
{
    sqlite3 *db;                                                                            // sqlite3 db struct
    sqlite3_stmt *stmt;
    char *ErrorMsg = NULL;
    int rc;
    char queryString[querylength];                                                          // final query string
    char querycopy[querylength];                                                            // copy of query string
    int i;                                                                                  // loop variable
    unsigned int num_rows;
    int debug;                                                                              // used for debugging

    debug = 0;                                                                              // set equal to 1 for debugging


    //----------------------------------------------------------------------------------------------------------------------------------------------------
    // construct final query string
    // SELECT * FROM ALMADB.RadTrans...

    // Debug:
    if (debug == 1){
        fprintf (stderr,"\n\nqueryString1 = %s\n",queryString1);
        fprintf (stderr,"queryString2 = %s\n",queryString2);
        fprintf (stderr,"querylength = %d\n",querylength);
        fprintf (stderr,"NumberOfMolecules = %i\n",NumberOfMolecules);
        if (NumberOfMolecules > 0){
            fprintf (stderr,"MoleculeNames[0] = %s\n", MoleculeNames[0]);
            fprintf (stderr,"MoleculeNames[NumberOfMolecules-1] = %s\n", MoleculeNames[NumberOfMolecules-1]);
        }
        fprintf (stderr,"dbName = %s\n", dbName);
    }
    if (NumberOfMolecules > 0){
        sprintf(queryString, "%s WHERE (%s = %s", queryString1, NameOfColumnName, MoleculeNames[0]);
        if (debug == 1){fprintf (stderr,">>> queryString = %s\n",queryString);}
        for (i = 1; i < NumberOfMolecules; i++)
	    {
            // Debug:
            if (debug == 1){fprintf (stderr,">>> %d\n",i);}

            sprintf(querycopy, "%s", queryString);
            sprintf(queryString, "%s OR %s = %s", querycopy, NameOfColumnName, MoleculeNames[i]);
            if (debug == 1){fprintf (stderr,">>> queryString = %s\n",queryString);}
	    }
        sprintf(querycopy, "%s", queryString);
        sprintf(queryString, "%s) %s", querycopy, queryString2);
    }else{
        sprintf(queryString, "%s %s", queryString1, queryString2);
    }

    // Debug:
    if (debug == 1){fprintf (stderr,"queryString = %s\n",queryString);}


    //----------------------------------------------------------------------------------------------------------------------------------------------------
    // connect to sqlite3 database
    rc = sqlite3_open(dbName, &db);
    if ( rc ){
        fprintf(stderr, "\n\n");
        fprintf(stderr, "subroutine numentr_sqlite3: Failed to connect to database: Error: %s\n", sqlite3_errmsg(db));
        fprintf (stderr,"\n\nqueryString = %s\n",queryString);
        *ok = 1;
        sqlite3_close(db);

        // Debug:
        if (debug == 1){fprintf (stderr,"rc = %i", rc);}

        /* check to see that the backend connection was successfully made */
    } else
    {

        // Debug:
        if (debug == 1){
            fprintf(stderr,"subroutine numentr_sqlite3: Connection to database '%s' succeeded.\n", dbName);
            fprintf (stderr,"rc = %i\n", rc);
        }
        *ok = 0;
    }


    //----------------------------------------------------------------------------------------------------------------------------------------------------
    // Execute the query

    // Debug:
    if (debug == 1){fprintf (stderr,"\nfinal queryString = '%s'\n", queryString);}

    rc = sqlite3_prepare_v2(db, queryString, -1, &stmt, NULL);
    if ( rc ){
        fprintf(stderr, "SQL error: %s\n", ErrorMsg);
        sqlite3_free(ErrorMsg);
        ErrorMsg=NULL;
        *ok = 1;
        sqlite3_close(db);

        // Debug:
        if (debug == 1){fprintf (stderr,"rc = %i\n", rc);}
    }


    // get number of entries
    while(1)
    {

        // fetch a row’s status
        rc = sqlite3_step(stmt);
        if(rc == SQLITE_ROW)
        {
            num_rows = sqlite3_column_int(stmt, 0);
            *number_lines = num_rows;
        }
        else if(rc == SQLITE_DONE)
        {

            // print number of entries to screen if allowed
            if (debug == 1){
                fprintf(stderr,"Number of entries of table:  num_rows = %i\n", num_rows);
            }
            break;
        }
    }


    // close connection to database
    sqlite3_finalize(stmt);
    sqlite3_close(db);                                                                      // close the database connection
}

